#!/usr/bin/env python3
import os, subprocess, tempfile, sys, time

AFTER_EFFECTS_APP = os.environ.get("AE_APP_NAME", "Adobe After Effects 2025")

# --- Paths ---
PROJECT     = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Baseball-Animation-2.aep"
CSV_PATH    = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/teams_rgb_correct.csv"
COMMON_JSX  = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/gl_common.jsxinc"
# This file name should match the JSX file below
JSX_BATCH   = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/batch_Homerun_H123.jsx"

# AE / template
COMP        = "Shatter-HR-384"
RS_TEMPLATE = "Best Settings"
OM_TEMPLATE = "PNG Sequence"

HOMEPLATE_DIR   = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/BaseBall/HomeRunAnimation_1/HomePlate"
LAYER_HOMEPLATE = "HomePlateRunNum"

# Output
OUTDIR        = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Homerun_Animation_2-384"
PATH_TEMPLATE = "{league}"

# League (CLI override optional)
LEAGUE = sys.argv[1] if len(sys.argv) > 1 else "MLB"

# Controls
AE_DEBUG          = "1"
PURGE_BEFORE      = "1"
NO_RENDER         = "0"
QUIT_APP          = "0"    # keep AE open across runs
TEXT_STROKE_WIDTH = "0.5"

# Logos
LOGO_DIR  = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Logos-Large/"
LOGO_TPL  = "{league}/{abbr}"
LOGO_EXTS = "png,jpg,jpeg,svg,ai,psd"

# Text/Shape layers to update inside precomp
INNER_LAYERS = "HOME,RUN"

def _write_bridge_jsx(env):
    def esc(s): return str(s).replace("\\","\\\\").replace('"','\\"')
    lines = [ '$.setenv("'+esc(k)+'","'+esc(v)+'");' for k,v in env.items() ]
    lines.append('$.evalFile(File("'+esc(JSX_BATCH)+'"));')
    code = "\n".join(lines) + "\n"
    tmp = tempfile.NamedTemporaryFile(prefix="ae_bridge_", suffix=".jsx", delete=False)
    tmp.write(code.encode("utf-8")); tmp.flush(); tmp.close()
    return tmp.name

def _osascript(script_text):
    proc = subprocess.run(["osascript", "-s", "o"], input=script_text, text=True, capture_output=True)
    out = (proc.stdout or "").strip()
    err = (proc.stderr or "").strip()
    if err:
        print("\n[osascript stderr]\n" + err, file=sys.stderr)
    proc.check_returncode()
    return out, err

def _ping_ae(app_name):
    try:
        _osascript(f'tell application "{app_name}" to return name')
        return True
    except subprocess.CalledProcessError:
        return False

def run_after_effects(env):
    os.makedirs(OUTDIR, exist_ok=True)
    bridge = _write_bridge_jsx(env)
    osa_src = f'''
set bridgePosix to "{bridge}"
set bridgeFile to POSIX file bridgePosix
with timeout of 86400 seconds
  tell application "{AFTER_EFFECTS_APP}"
    activate
    try
      set res to DoScriptFile bridgeFile
      return "AERUN:OK:" & (res as text)
    on error errMsg number errNum
      return "AERUN:ERR:" & errNum & ":" & errMsg
    end try
  end tell
end timeout
'''
    _osascript(osa_src)

def make_env(anim_name, homeplate_path, runnum):
    return {
        # core
        "AE_PROJECT": PROJECT,
        "AE_CSV": CSV_PATH,
        "AE_COMP": COMP,
        "AE_COMMON_JSX": COMMON_JSX,

        # layer names
        "AE_LAYER_HOMERUN": "HOMERUN",
        "AE_LAYER_HR_PRECOMP": "HomeRun PreComp",
        "AE_LAYER_LOGO": "TeamLogo",
        "AE_LAYER_TEAMNAME": "TeamName",
        "AE_INNER_STROKE_NAMES": INNER_LAYERS,
        "AE_TEXT_STROKE_WIDTH": TEXT_STROKE_WIDTH,

        # per-run plate swap
        "AE_LAYER_HOMEPLATE": LAYER_HOMEPLATE,
        "AE_HOMEPLATE_PATH": homeplate_path,
        "AE_RUNNUM": str(runnum),

        # logos
        "AE_LOGO_DIR": LOGO_DIR,
        "AE_LOGO_PATH_TEMPLATE": LOGO_TPL,
        "AE_LOGO_EXTS": LOGO_EXTS,

        # output & templates
        "AE_OUTDIR": OUTDIR,
        "AE_PATH_TEMPLATE": PATH_TEMPLATE,
        "AE_ANIM": anim_name,
        "AE_RS_TEMPLATE": RS_TEMPLATE,
        "AE_OM_TEMPLATE": OM_TEMPLATE,

        # controls
        "AE_PURGE_BEFORE_RENDER": PURGE_BEFORE,
        "AE_NO_RENDER": NO_RENDER,
        "AE_QUIT": QUIT_APP,
        "AE_LEAGUE": LEAGUE,
        "AE_DEBUG": AE_DEBUG,
    }

if __name__ == "__main__":
    print("=== After Effects Homerun Animation Batch v2 ===")
    if not _ping_ae(AFTER_EFFECTS_APP):
        print(f'!! Could not talk to "{AFTER_EFFECTS_APP}"', file=sys.stderr)
        sys.exit(2)

    # RunNum 1..4
    for runnum in range(1, 5):
        anim_name = f"H-{runnum}_2_3"
        homeplate_path = os.path.join(HOMEPLATE_DIR, f"HomePlate-{runnum}.png")
        if not os.path.isfile(homeplate_path):
            print(f"!! Missing HomePlate image: {homeplate_path}", file=sys.stderr)
        print(f"\n== RunNum {runnum} :: AE_ANIM = {anim_name} :: HomePlate = {homeplate_path}")
        run_after_effects(make_env(anim_name, homeplate_path, runnum))
        time.sleep(2)  # small breather for AE
    print("\nDone.")
